/*
 *  uhtget
 *  Simple http-protocol file retriever.
 *
 *  Copyright (C) 2000 Jens Laas jens.laas@data.slu.se
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 * Changes:
 * v 0.1 First release
 * v 0.2 2000-10-16 first bugfix in cat_int()
 * v 0.3 2000-10-16 support for freer format in /etc/resolv.conf
 * v 0.4 2001-01-11 name change from htget to uhtget. Makefile included.
 * v 0.5 2002-10-14 fixed memory allocation buglet. Reported by David Dueck.
 */

/*
  Functionality:
  o  limited :-)
  o  accepts ip-addresses as numbers and DNS-names.
  o  expects a MIME-type header from the http server.

  Usage:
  o  uhtget [-h] [-f] [--] SERVER[:PORT][/path/to/file]
  o  -h         help
  o  -f         force, dont prompt when overwriting.
  o  --         output to stdout not file
  
  Compiling:
  o  make. Makefile may need modifying.
  
*/

/* Accepts tabs and spaces as separators (in /etc/resolv.conf) if defined. */
#define RESOLVE_NICER

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/poll.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include <stdarg.h>

#define BSIZE 512

char error_string[BSIZE];

char url[BSIZE];
char server[BSIZE];
char file[BSIZE];
int port = 80;
int config_force = 0;
int config_stdout = 0;

char out_str[512];

/*
  Output simplifying functions:
  init_io()
  cat(char *)
  cat_int(int )
  print( char * || file_descriptor)
*/

void init_io()
{
  out_str[0] = 0;
}

void cat(char *s)
{
  strcpy(&out_str[ strlen(out_str) ], s);
}

void vcat( int n, ... )
{
  va_list ap;
  
  va_start( ap, n);
  while(n--)
    strcpy(&out_str[ strlen(out_str) ], va_arg(ap, char *));
  va_end(ap);
}

void cat_int(int nr)
{
  char num[32], *n, *o;

  n = num;

  do
    {
      *n++ = '0' + nr % 10;
      nr = nr / 10;
    }
  while( nr > 0 );

  *n = 0;
  o = out_str + strlen(out_str);
  for(--n; n >= num; n--)
    *o++ = *n;
  *o = 0;
}

  
void print(char *s)
{
  if((unsigned long) s>256)
    strcpy(s, out_str);
  else
    {
      if(s == 0)
	s = (char *) 1;
      write((int)s, out_str, strlen(out_str));
    }
  out_str[0] = 0;
}

/*
  String handling functions:
  my_strrchr( char *s, int chr)
  my_strncmp( char *s1, char *s2, int n)
  read_str(int size, char *buffer)
*/

char *my_strrchr( const char *s, int c)
{
  char *p = (char *) s + strlen(s);

  for( ; p >= s; --p)
    if(*p == c)
      return p;
  
  return 0;
}

int my_strncmp(char *s1, char *s2, size_t n)
{
#if 0
  if( (strlen(s1)<n) || (strlen(s2)<n) )
    return 1;
#endif
  while(--n)
    if(*s1++ != *s2++)
      return 1;
  return 0;
}

int read_str(int n, char *b)
{
  int g,got = 0;
  char *p;

  *b = 0;

  do
    {
      if(got == n)
	return 0;
      
      g = read(0, b+got, 1);
      if(g <= 0)
	return 0;
      got += g;
      p = strchr(b, '\n');
    }
  while(!p);
  
  return 1;
}

/*
  Simplified version of inet_aton()
*/

int my_inet_aton_base( char *host )
{
  char *p = host;
  int a,b,c,d;

  a = atoi(p);
  
  if(!(*p >= '0' && *p <= '9'))
    return 0;
  
  p = strchr(p, '.');
  if(!p++)
    return 0;
  b = atoi(p);

  p = strchr(p, '.');
  if(!p++)
    return 0;
  c = atoi(p);

  p = strchr(p, '.');
  if(!p++)
    return 0;
  d = atoi(p);
  
#if 0
  adr->s_addr = (a<<24) + (b<<16) + (c<<8) +d;
#endif
  /* Not sure about this.
     Shouldnt there be a htonl() somewhere around here.. */
  return (d<<24) + (c<<16) + (b<<8) +a;
}

int my_inet_aton( char *host, struct in_addr *adr )
{
  adr->s_addr = my_inet_aton_base(host);
  
  if(adr->s_addr)
    return 1;
  return 0;
}

/*
  UDP-functions
*/

int udp_new_sock( int port )
{
  int one = 1;
  int s;
  struct sockaddr_in my_addr;
  
  memset( &my_addr, 0, sizeof(my_addr));
  
  s = socket( PF_INET, SOCK_DGRAM, IPPROTO_UDP); // udp = 17
  if(s<0)
    return -1;
  
  setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (char *)&one, sizeof(one));
  
  my_addr.sin_family = AF_INET;
  my_addr.sin_port = htons(port);
  
  return s;
}

int udp_recv( int sock,
	      void *buffer,
	      int buff_size)
{
  struct sockaddr_in from_addr;
  int fromlen;
  int got;
  
  fromlen = sizeof( from_addr );
  
  got = recvfrom( sock, buffer, buff_size, 0, (struct sockaddr*)&from_addr, &fromlen);
  if(got<1)
    return -1; /* error */

#if 0  
  if(f_port)
    {
      *f_port = ntohs(from_addr.sin_port);
    }
  if(f_addr)
    {
      *f_addr = (uint) from_addr.sin_addr.s_addr;
      // *.s_addr is 4 bytes with inet adr = x.x.x.x
    }
#endif
  return got;
}

int udp_send( int sock,
	      void *data,
	      int len,
	      unsigned int to,
	      int port )
{
  struct sockaddr_in to_addr;
  int tolen;
  int got;
  
  tolen = sizeof( to_addr );
  memset( &to_addr, 0, tolen);
  
  to_addr.sin_port = htons( port );
  to_addr.sin_family = AF_INET;
  //to_addr.sin_addr.s_addr = htonl( 130<<24 | 238<<16 | 98<<8 | 46 );
  //to_addr.sin_addr.s_addr = htonl( 127 << 24 | 1 );
  //to_addr.sin_addr.s_addr = htonl( to );
  to_addr.sin_addr.s_addr = to;
  
  got = sendto( sock, data, len, 0, (struct sockaddr*)&to_addr, tolen);
  if(got<1)
    return -1;
  return got;
}

/*
  DNS lookup
*/

int dns_lookup(char *host, struct in_addr *adr )
{
  unsigned char *p, *start, *c;
  char hcopy[BSIZE];
  int hlen, answers;
  unsigned char send_b[BSIZE+1], recv_b[BSIZE+1];
  char nameserv[BSIZE];
  unsigned int nameserv_ip;
  struct pollfd pollfd;
  int result, s, count = 15;
  int id = 33445;
  struct timeval time_now;

  /* randomize ID with time */
  gettimeofday(&time_now, 0);
  id += time_now.tv_usec & 0x7ff;

  /* Generate QNAME field */
  start = hcopy;
  for(p=host, c=hcopy+1; *p; p++)
    {
      if(*p == '.')
	{
	  *start = c-start-1;
	  start = c;
	  c++;
	}
      else
	*c++ = *p;
    }
  *start = c-start-1;
  *c++ = 0;
  
  /* find nameserver address
     in /etc/resolv.conf */
  {
    int fd;
    fd = open( "/etc/resolv.conf", O_RDONLY);
    if(fd<0)
      {
	cat("Failed open /etc/resolv.conf\n");
	print(0);
	return 0;
      }
    nameserv[0] = 0;
    p = recv_b;
    while(1 == read(fd, p, 1))
      {
	if(*p == '\n')
	  {
	    *p=0;
	    if(!my_strncmp(recv_b, "nameserver", 10))
	      {
#ifdef RESOLVE_NICER
		for(p=recv_b+10;*p;p++)
		  if((*p != ' ')&&(*p != '\t'))
		    break;
		strcpy(nameserv, p);
#else
		strcpy(nameserv, &recv_b[11]);
#endif
		break;
	      }
	    else
	      p = recv_b;
	  }
	else
	  p++;
      }
  }
  
  nameserv_ip = my_inet_aton_base(nameserv);
  if(!nameserv_ip)
    {
      cat("No nameserver in /etc/resolv.conf\n");
      print(0);
      return 0;
    }
  
  s = udp_new_sock( 0 );
  
  /* Send query to nameserver */
 query:
  /* ID */
  p = send_b;
  *p++ = (id >> 8) & 255;
  *p++ = id & 255;

  memcpy(p, "\1\0\0\1\0\0\0\0\0\0", 10);
  p += 10;
  
#if 0
  /* RD */
  *p++ = 1;

  /* RCODE */
  *p++ = 0;

  /* QDCOUNT */
  *p++ = 0;
  *p++ = 1;

  /* ANCOUNT */
  *p++ = 0;
  *p++ = 0;

  /* NSCOUNT */
  *p++ = 0;
  *p++ = 0;

  /* ARCOUNT */
  *p++ = 0;
  *p++ = 0;
#endif

  /* QNAME. represented by series of labels.
   The name: my.domain.org is split into:
   2my, 6domain, 3org, 0
  */
  
  for(c=hcopy; *c; c++)
    *p++ = *c;
  *p++ = 0; /* Null terminate */

  /* QTYPE */
  *p++ = 0;
  *p++ = 1; /* A */
  *p++ = 0;
  *p++ = 1; /* IN, the internet */

  if(!count--)
    return 0;

  hlen = p - send_b;
  if(udp_send( s, send_b, hlen, nameserv_ip, 53) < 0)
    goto query;
  
  /* wait for reply with poll */
  pollfd.fd = s;
  pollfd.events = POLLIN;
  pollfd.revents = 0;
  result = poll( &pollfd, 1, 300); /* 300 ms timeout */
  
  /* timeout -> resend */
  if(!result)
    goto query;
  if(!(pollfd.revents & POLLIN))
    goto query;
  
  if( udp_recv(s,recv_b,BSIZE) < 1)
    goto query;
  
  /* Got an answer */

  if(! ((recv_b[0] == send_b[0]) &&  (recv_b[1] == send_b[1])))
    goto query; /* answer to wrong question */
  //printf("ID ok\n");
  
  answers = (recv_b[6] << 8) + recv_b[7];
  if(!answers)
    return 0; /* no answer records */
  
  //printf("We got %d answers\n", answers);
  
  p = recv_b+hlen; /* point to answer field */
  
  while(answers--)
    {
      int fail = 0;
      
      //printf("Stepping past name field\n");
      /* Step past NAME field, Ignore content.. */
      while(*p)
	{
	  if((unsigned char)*p > 63)
	    {
	      p++;
	      break;
	    }
	  else
	    count = (int) *p++;
	  while(count--)
	    p++;
	}
      p++;
      //printf("Passed NAME\n");
      
      /* TYPE(2) CLASS(2) TTL(4) RDLENGTH(2) */
      if(*p++ != 0)
	fail = 1; /* cannot by A */
      if(*p++ != 1)
	fail = 1; /* definitely not A */
      //printf("Passed TYPE\n");
      if(*p++ != 0)
	fail = 1; /* cannot by IN */
      if(*p++ != 1)
	fail = 1; /* definitely not IN */
      //printf("Passed CLASS\n");
      p+=4; /* step past TTL */

      if(fail)
	{
	  uint step;
	  
	  step = (uint) *p++ << 8;
	  step += (uint) *p++;
	  p += step;
	  continue;
	}
      
      p+=2; /* step past rdlength */
      
      /* RDATA */
      
      adr->s_addr = ( ((uint)*p++)<<24 );
      adr->s_addr += ( ((uint)*p++)<<16 ); 
      adr->s_addr += ( ((uint)*p++)<<8 );
      adr->s_addr += (uint) *p;
      adr->s_addr = htonl(adr->s_addr);
      return 1;
    }
  return 0;
}

int do_connect(char *host, int port)
{
  struct sockaddr_in address;
  int err, newtcp_fd;
  
  memset((char *)&address, 0, sizeof(address));
  
  if( !my_inet_aton( host, &address.sin_addr ) )
    {	
      if(!dns_lookup( host, &address.sin_addr ))
	{
	  strcpy(error_string, "Failed to resolve address.\n");
	  return -1;
	}
    }
  
  address.sin_family = AF_INET;  
  address.sin_port = htons(port);
  
  newtcp_fd = socket(address.sin_family, SOCK_STREAM, 0);
   if (newtcp_fd < 0) 
     {
       /* error creating socket */
	strcpy(error_string, "Error create socket!\n");
	return -1;
     }
   
   err = connect(newtcp_fd, (struct sockaddr *)&address, sizeof(address));
   
   if (err < 0)
     {
       strcpy(error_string,  "Failed to connect\n");
       return -1;
     }
   
   return newtcp_fd;
}

/*
  Issue a http-request over the given socket.
  Save result to given filename.
*/

int get(int s, char *file, char *host)
{
  char b[BSIZE];
  char local[BSIZE];
  char answer[BSIZE];
  char *p, last;
  int f = 1;
  int got;
  
  if(config_stdout)
    goto download;
  
  strcpy(answer, "no");
  
  if(strlen(file) == 1)
    strcpy(local, "download");
  else
    {
      p = my_strrchr(file, '/');
      if(p)
	{
	  if(strlen(p) == 1)
	    strcpy(local, "download");
	  else
	    strcpy(local, p+1);
	}
      else
	strcpy(local, file);
    }
  
  if( (f = open(local, O_RDONLY)) > -1)
    {
      close(f);
      if(config_force)
	goto remove;
      vcat(3, "File \"", local, "\" exists.\n Delete (y/n)? ");
      print((char *)2);
      
      got = read_str(5, answer);
      if(got == 1 && answer[0] == 'y')
	{
	remove:
	  unlink(local);
	}
    }
  
  f = open(local, O_WRONLY | O_CREAT | O_EXCL, S_IRUSR|S_IWUSR);
  if(f<0)
    {
      vcat(3, "Failed to open local file ", local, ".\n");
      print( error_string );
      return -1;
    }
  
  
  vcat(3, "Local file: \"", local, "\".\n");
  print((char *)2);
  
 download:
  vcat(5, "GET ", file,
       " HTTP/1.0\r\nHost:", host,
       "\r\nConnection: close\r\n\r\n");
  print((char *)s);
  
  //write(s, b, strlen(b));
  
  /* read until 2 newline read */
  /* Change code here if you want to accept answers from
     http-server without a content-header */
  
  last = 0;
  
  while( (got = read(s, b, 1)) > 0)
    {
      if(b[0] == '\n' && (last == '\n'))
	break;
      if(b[0] == '\r')
	continue;
      last = b[0];
    }
  
  if(got != 1)
    {
      strcpy(error_string, "Failed to read header");
      close(f);
      unlink(local);
      return -1;
    }
  
  /* read the file */
  while( (got = read(s, b, BSIZE)) > 0)
    {
      write(f, b, got);
    }
  
  close(f);
  return 0;
}

int main(int argc, char *argv[])
{
  char *p;
  int s, status;

  init_io();
  
  strcpy(error_string, "Unknown error");
  
  while( (argc > 1) && argv[1][0] == '-')
    {
      if(argv[1][1] == 'h')
	goto usage;
      if(argv[1][1] == 'f')
	config_force = 1;
      if(argv[1][1] == '-')
	config_stdout = 1;
      argc--;
      argv++;
    }
  
  if(argc != 2)
    {
    usage:
      vcat(3, "Usage:\n", argv[0], " [-h] [-f] [--] url\n");
      print((char *)2);
      exit(1);
    }

  if(strlen(argv[1]) >= BSIZE)
    argv[1][BSIZE] = 0;
  
  if(my_strncmp(argv[1], "http://", 7))
    strcpy(url, argv[1]);
  else
    strcpy(url, (argv[1])+7);


  strcpy(server, url);
  p = strchr(server, '/');
  if(p)
    *p = 0;
  
  p = strchr(url, '/');
  if(p)
    strcpy(file, p);
  else
    strcpy(file, "/");
  
  p = strchr(server, ':');
  if(p)
    {
      port = atoi(p+1);
      *p=0;
    }
  
  vcat(2, "Connecting to ", server);
  
  if(port != 80)
    {
      cat(" at ");
      cat_int( port );
    }
  
  cat(".\n");
  print((char *)2);

  s = do_connect(server, port);
  
  if(s < 0)
    goto error;
  
  vcat(3, "Retrieving ", file, ".\n");
  print((char *)2);
  status = get(s,file, server);
  
  if(status<0)
    {
    error:
      vcat(3, "Error: ", error_string, "\n");
      print((char *)2);
      exit(1);
    }
  
  close(s);
  return 0;
}
