/*  Monte-Carlo simulation code for statistical physics
    Copyright (C) 2001-2004  Sylvain Reynal
    
    Dpartement de Physique
    Ecole Nationale Suprieure de l'Electronique et de ses Applications (ENSEA)
    6, avenue du Ponceau,    
    F-95014 CERGY CEDEX

    et
    
    Laboratoire de Physique Thorique et Modlisation (LPTM)
    Universit de Cergy-Pontoise - Site de Neuville
    F-95031 CERGY CEDEX
    
	
    Tel : 00 +33 130 736 245
    Fax : 00 +33 130 736 667
    e-mail : reynal@ensea.fr
    web page : http://www.ensea.fr/staff/reynal/
*/

package fr.ensea.montecarlo.data;

import fr.ensea.Version;
import fr.ensea.Localizer;
import fr.ensea.chart.*;
import fr.ensea.montecarlo.*;
import fr.ensea.montecarlo.multicanonical.*;
import fr.ensea.montecarlo.data.*;
import fr.ensea.montecarlo.model.*;
import fr.ensea.montecarlo.misc.*;

import java.awt.*;
import java.awt.event.*;
import java.applet.Applet;
import java.text.NumberFormat;
import java.util.Locale;

/**
 * Holds several "Samples" together, e.g. the energy, magnetization, etc...
 */
public class SamplesBag implements ActionListener, AdjustmentListener {
	Samples energySamples; 
	Samples magnetizationSamples;
	Samples energyRGSamples;  // not always used
	Plot2D graphSamples, graphHistogram, graphDOS;
	int graphUpdateIdx = 0, graphUpdateDelay = 10;

	// --- UI ---
	AbstractApplet controller;
	Label acceptanceLBL,autocorrelationLBL,graphUpdateDelayLBL;
	Button resetGraphsBUT;
	Scrollbar graphUpdateDelaySB;
	NumberFormat nfDouble;

	public SamplesBag(int defMCTherm){
		energySamples = new Samples();
		magnetizationSamples = new Samples();
		energySamples.setThermalizationTime(defMCTherm);
		magnetizationSamples.setThermalizationTime(defMCTherm);
		magnetizationSamples.setAutocomputeCorrelationTime(true);
		// not always used
		energyRGSamples = new Samples();
		energyRGSamples.setThermalizationTime(defMCTherm);
	}
	
	// --- properties ---
	public Samples getEnergySamples(){
		return energySamples;
	}
	
	public Samples getMagnetizationSamples(){
		return magnetizationSamples;
	}
	
	public Samples getEnergyRGSamples(){
		return energyRGSamples;
	}
	
	public int size(){
		return energySamples.size();
	}
	
	public boolean isEmpty(){
		return energySamples.isEmpty();
	}
	
	public void clear(){
		energySamples.clear();
		energyRGSamples.clear();
		magnetizationSamples.clear();
		if (controller instanceof WLApplet){
			WLDOSHistogram histo = ((WLSimulationThread)controller.getSimulationThread()).getWLDOSHistogram();
			histo.reset();
		}
	}
	
	public double getCorrelationTime(){
		return magnetizationSamples.getCorrelationTime();
	}		
		
	public void setMCSWait(int i){
		energySamples.setMCSWait(i);
		energyRGSamples.setMCSWait(i);
		magnetizationSamples.setMCSWait(i); 
	}
	
	public int getMCSWait(){
		return energySamples.getMCSWait();
	}
	
	public void setThermalizationTime(int i){
		energySamples.setThermalizationTime(i);
		energyRGSamples.setThermalizationTime(i);
		magnetizationSamples.setThermalizationTime(i);
	}
	
	public int getThermalizationTime(){
		return energySamples.getThermalizationTime();
	}
		
	// --- samples accumulation ---
	public void addEnergySample(double E){
		energySamples.addSample(E);
	}
	
	public void addMagnetizationSample(double M){
		magnetizationSamples.addSample(M);
	}

	public void addEnergyRGSample(double E){
		energyRGSamples.addSample(E);
	}
	
	// --- thermal averages ---
	public double computeEnergyMean(){
		return energySamples.computeMean();
	}
	
	public double computeEnergyRGMean(){
		return energyRGSamples.computeMean();
	}

	public double computeMagnetizationMean(){
		return magnetizationSamples.computeMean();
	}
	
	public double computeEnergyVariance(double meanEnergy){
		return energySamples.computeVariance(meanEnergy);
	}
	
	public double computeMagnetizationVariance(double meanMagnetization){
		return magnetizationSamples.computeVariance(meanMagnetization);
	}
	
	// --- UI ---

	public void updateGUI(){
		graphUpdateIdx++;
		if (graphUpdateIdx > graphUpdateDelaySB.getValue()){
			acceptanceLBL.setText(controller.getSimulationThread().getCountersRateLabel());
			autocorrelationLBL.setText(Localizer.get("Autocorrelation")+"="+nfDouble.format(getCorrelationTime())+" "+Localizer.get("steps"));
			graphSamples.dataChanged();
			if (graphHistogram!=null) graphHistogram.dataChanged();
			if (graphDOS!=null) graphDOS.dataChanged();
			graphUpdateIdx=0;
		}
	}

	/**
	 * Create the left-most pane containing graphs of sample measurements  
	 */
	public Panel createUI(AbstractApplet controller){
		this.controller = controller;
		Dimension graphPreferredSize = new Dimension(controller.getSize().width/4, controller.getSize().height/5);
		nfDouble = NumberFormat.getNumberInstance(Locale.US);
		nfDouble.setGroupingUsed(false);
		nfDouble.setMaximumFractionDigits(2);

		// --- Pane 1 = acceptance rate ---
		Panel controlPane1 = new Panel();
		controlPane1.add(acceptanceLBL=new Label());

		// --- Pane 2 = autocorrelation time ---
		Panel controlPane2 = new Panel();
		controlPane2.add(autocorrelationLBL = new Label());

		// --- Pane 3 = reset graphs, etc. ---
		Panel controlPane3 = new Panel();
		controlPane3.add(graphUpdateDelayLBL=new Label(Localizer.get("Refresh")+" "+Integer.toString(graphUpdateDelay)+ Localizer.get("steps")));
		controlPane3.add(graphUpdateDelaySB = new Scrollbar(Scrollbar.HORIZONTAL, graphUpdateDelay, 5, 0, 1005));
		controlPane3.add(resetGraphsBUT=new Button(Localizer.get("Clear")));

		// --- Pane "graphs" = graphs of E and M (and possibly E(RG)) ---
		graphSamples = new Plot2D(AbstractApplet.PANELS_BACKGROUND, AbstractApplet.PLOT2D_FRAME_COLOR1, AbstractApplet.PLOT2D_GRID_COLOR);
		String magLbl;
		if (controller.getLattice() instanceof AFIsingLattice) magLbl = "M(stag)";
		else magLbl = "M";
		graphSamples.addCurve(getMagnetizationSamples(), magLbl, AbstractApplet.PLOT2D_CURVE_COLOR2, DecoratedCurve2D.LINES);
		graphSamples.addCurve(getEnergySamples(), "E", AbstractApplet.PLOT2D_CURVE_COLOR1, DecoratedCurve2D.LINES);
		boolean displayERG = controller instanceof MCApplet && ((MCApplet)controller).isRGActive(); 
		if (displayERG) 
			graphSamples.addCurve(getEnergyRGSamples(), "E(RG)", AbstractApplet.PLOT2D_CURVE_COLOR2, DecoratedCurve2D.LINES);
		
		graphSamples.setForeground(Color.white); // title, keys
		graphSamples.setTitle("");
		graphSamples.setPreferredSize(graphPreferredSize);

		// --- Pane "energy histogram" ---
		boolean displayHistogram = (controller instanceof MCApplet && ((MCApplet)controller).isRGActive()==false);
		displayHistogram = displayHistogram || (controller instanceof WLApplet);
		if (displayHistogram){
			this.graphHistogram = new Plot2D(AbstractApplet.PANELS_BACKGROUND, AbstractApplet.PLOT2D_FRAME_COLOR1, AbstractApplet.PLOT2D_GRID_COLOR);
			//if (lattice instanceof AFIsingLattice) graphHistogram.xAxis().setManualAxisBounds(-0.1,2.1);
			//else graphHistogram.xAxis().setManualAxisBounds(-2.1,0.1);
			if (controller instanceof MCApplet){
				Histogram histo = getEnergySamples().createHistogram(50); // 50 bins
				graphHistogram.addCurve(histo, "", AbstractApplet.PLOT2D_CURVE_COLOR1, DecoratedCurve2D.SAMPLES);
			}
			else if (controller instanceof WLApplet){
				WLDOSHistogram histo = ((WLSimulationThread)controller.getSimulationThread()).getWLDOSHistogram();
				Curve2D histoCurve = histo.createHistoCurve2D(controller.getLattice());
				graphHistogram.addCurve(histoCurve, "", AbstractApplet.PLOT2D_CURVE_COLOR1, DecoratedCurve2D.SAMPLES);
			}
			graphHistogram.setForeground(Color.white); // title, keys
			graphHistogram.setTitle(Localizer.get("EnergyHistogram"));
			graphHistogram.setPreferredSize(graphPreferredSize);
		}

		// --- Pane "density of state" (WL only) ---
		boolean displayDOS = (controller instanceof WLApplet);
		if (displayDOS){
			this.graphDOS = new Plot2D(AbstractApplet.PANELS_BACKGROUND, AbstractApplet.PLOT2D_FRAME_COLOR1, AbstractApplet.PLOT2D_GRID_COLOR);
			WLDOSHistogram histo = ((WLSimulationThread)controller.getSimulationThread()).getWLDOSHistogram();
			Curve2D dosCurve = histo.createSECurve2D(controller.getLattice());
			graphDOS.addCurve(dosCurve, "", AbstractApplet.PLOT2D_FRAME_COLOR2, DecoratedCurve2D.DOTS);
			graphDOS.setForeground(Color.white); // title, keys
			graphDOS.setTitle("S(E)=ln n(E)");
			graphDOS.setPreferredSize(graphPreferredSize);
		}
			
			

		// layout:
		GridBagLayout gridbag = new GridBagLayout();
		GridBagConstraints c = new GridBagConstraints();
		c.insets = new Insets(5,5,5,5);
		Panel graphPanel = new Panel(gridbag);
		c.fill = GridBagConstraints.BOTH;
		c.gridwidth = GridBagConstraints.REMAINDER;
		c.weightx = 1.0;

		if (controller instanceof MCApplet){
			c.weighty =4;
			gridbag.setConstraints(controlPane1,c);
			graphPanel.add(controlPane1);
			gridbag.setConstraints(controlPane2,c);
			graphPanel.add(controlPane2);
		}

		c.weighty = 10.0;
		gridbag.setConstraints(graphSamples,c);
		graphPanel.add(graphSamples);

		if (displayHistogram) {
			c.weighty = 10.0;
			gridbag.setConstraints(graphHistogram,c);
			graphPanel.add(graphHistogram);
		}

		if (displayDOS){
			c.weighty = 10.0;
			gridbag.setConstraints(graphDOS,c);
			graphPanel.add(graphDOS);
		}
			
		c.weighty = 5;
		gridbag.setConstraints(controlPane3,c);
		graphPanel.add(controlPane3);

		if (displayERG) {
			c.weighty = 20.0;
			c.gridheight=GridBagConstraints.REMAINDER;
			gridbag.setConstraints(((MCApplet)controller).getRenormalizationPanel(),c);
			graphPanel.add(((MCApplet)controller).getRenormalizationPanel());
		}

		// --- listeners ---
		resetGraphsBUT.addActionListener(this);
		graphUpdateDelaySB.addAdjustmentListener(this);
		
		graphSamples.addMouseListener(new GraphMouseHandler(controller,this.graphSamples));
		graphSamples.addMouseMotionListener(new StatusHelpMouseHandler(controller,graphSamples.getTitle()+": "+Localizer.get("HelpStatus1")));
		if (graphHistogram!=null) {
			graphHistogram.addMouseListener(new GraphMouseHandler(controller,this.graphHistogram));
			graphHistogram.addMouseMotionListener(new StatusHelpMouseHandler(controller,graphHistogram.getTitle()+": "+Localizer.get("HelpStatus1")));
		}
		if (graphDOS!=null) {
			graphDOS.addMouseListener(new GraphMouseHandler(controller,this.graphDOS));
			graphDOS.addMouseMotionListener(new StatusHelpMouseHandler(controller,graphDOS.getTitle()+": "+Localizer.get("HelpStatus1")));
		}
		
		// --- other init
		graphUpdateIdx = graphUpdateDelaySB.getValue();
		
		return graphPanel;
	}
	
	public void resetSamplesGraphs(){
		controller.getSimulationThread().reinitCounters();
		clear();
		graphSamples.dataChanged();
		if (graphHistogram!=null) graphHistogram.dataChanged();
	}


	public void actionPerformed(ActionEvent evt){
		if (evt.getSource() == resetGraphsBUT){
			resetSamplesGraphs();
		}
	}
	
	public void adjustmentValueChanged(AdjustmentEvent evt){
		if (evt.getSource() == graphUpdateDelaySB){
			graphUpdateDelay = graphUpdateDelaySB.getValue();
			graphUpdateDelayLBL.setText(Localizer.get("Refresh")+". "+Integer.toString(graphUpdateDelay)+ Localizer.get("steps"));
		}
	}
		
}
