/*  Monte-Carlo simulation code for statistical physics
    Copyright (C) 2001-2004  Sylvain Reynal
    
    Dpartement de Physique
    Ecole Nationale Suprieure de l'Electronique et de ses Applications (ENSEA)
    6, avenue du Ponceau,    
    F-95014 CERGY CEDEX

    et
    
    Laboratoire de Physique Thorique et Modlisation (LPTM)
    Universit de Cergy-Pontoise - Site de Neuville
    F-95031 CERGY CEDEX
    
	
    Tel : 00 +33 130 736 245
    Fax : 00 +33 130 736 667
    e-mail : reynal@ensea.fr
    web page : http://www.ensea.fr/staff/reynal/
*/

package fr.ensea.montecarlo.canonical;

import java.io.*;
import java.awt.Color;
import java.text.*;

import fr.ensea.Localizer;
import fr.ensea.montecarlo.data.*;
import fr.ensea.montecarlo.model.*;
import fr.ensea.montecarlo.canonical.*;
import fr.ensea.montecarlo.misc.*;


/**
 * Algorithme de Metropolis. Class abstraite pour des modles de spins quelconques.
 */
public abstract class AbstractMetropolis extends Montecarlo  {

	protected long attemptedMoves; // nombre de mises--jour tentes
	protected long acceptedMoves; // nombre de mises--jour acceptes

	/**
	 * Initialise un nouvel algorithm de Metropolis
	 * @param energySamples liste pour les mesures de l'nergie
	 * @param magnetizationSamples liste pour les mesures de l'aimantation
	 * @param kT la temprature par dfaut
	 */
	public AbstractMetropolis(SamplesBag bag, double kT) {
		super(bag, kT);
	}

	/**
	 * Modifie la temprature courante et remet les compteurs  zro.
	 */
	public synchronized void setTemperature(double kT){
		this.kT = kT;
		reinitCounters();
	}
	
	/**
	 * Gnre un nouveau nombre alatoire distribu uniformment dans l'intervalle [0,1), et le retourne.
	 */
	public double createRandomNumber(){
		return Math.random();
	}

	/**
	 * Ralise un sweep de l'ensemble du rseau
	 */
	public abstract void sweep();

	///////////////////////////////////////////////////////////
	/// Statistiques
	///////////////////////////////////////////////////////////

	/**
	 * Retourne le taux de mise--jour russies en pourcents.
	 */
	public double getAcceptanceRate(){
		if (attemptedMoves==0) return 0;
		return (double)acceptedMoves/(double)attemptedMoves;
	}

	/**
	 * Rinitialise les compteurs de mises--jours acceptes
	 */
	public synchronized void reinitCounters(){
		this.acceptedMoves = this.attemptedMoves = 0;
	}
	
	public String getCountersRateJLabel(){
		return Localizer.get("AcceptedMoves")+"="+fr.ensea.montecarlo.AbstractApplet.formatPercent(getAcceptanceRate());
	}		
	
	
	/////////////////////////////////////////////////////////////////////////////////
	// TEST
	/////////////////////////////////////////////////////////////////////////////////

	/**
	 * Retourne un message d'information
	 */
	public String toString(){
		String s = "Algorithm de Metropolis :\n";
		s += "\ttaux de mises--jour russies = " + Double.toString(getAcceptanceRate()) + "%";
		return s;
	}

	/////////////////////////////////////////////////////////////////////////////////
	// TEST
	/////////////////////////////////////////////////////////////////////////////////

}

