package Convolution;
import javax.swing.*;
import javax.swing.border.BevelBorder;
import javax.swing.event.MouseInputListener;
import java.awt.*;
import java.awt.event.*;
import java.util.Random;

/**
 * Diese Klasse realisiert verschiedene Faltungsoperationen,wie
 * den Sobel-,Roberts-und LaPlace-Operator.Ausserdem besteht die
 * Moeglichkeit einen eigenen Faltungskern zu definieren.Das Eingangs-
 * bild kann beliebig mit gausschem Rauschen oder 'Salt and Pepper' versehen
 * werden.
 * Die Oberflaeche ist durch zwei Panels aufgebaut. Das erste ist das 
 * Operationpanel,in dem die Faltungsoperationen und die Bilder ausge-
 * waehlt werden koennen. Das zweite Panel beinhaltet die Bilder selbst,
 * also Eingangs-und Ausgangsbild. 
 * 
 * @author Rene Iser, Behrang Karimibabak, Simon Winkelbach
 *
 */

public class ConvolutionApplet extends JApplet implements MouseInputListener{

    protected iRPImage inputImage,outputImage; //Eingangs-und Ausgangsbild
    protected ImageWindow input,output;
    protected ConvolutionViewer viewer; //Panel zur Darstellung der Bilder
    protected JPanel operationPanel,messagePanel;
    protected JPanel[] toolMainPanel,toolSubPanel;
    protected JDialog pictureDialog,toolDialog;
    protected boolean sobel,roberts,laPlace,userDef; //Definiert welcher Operator aktiv ist
 
    protected JLabel sobelLabel;
    protected JLabel message1,message2,robertsLabel,laPlaceLabel,userDefLabel;
    protected JButton choosePicture,chooseTool;
    protected JButton convolve; //Start der eigens definierten Faltung
    protected JButton addGaus;//Hinzufuegen von Gausschem Rauschen
    protected JButton saltPepper;//Hinzufuegen von salt & Pepper
    protected JButton sobelButton,robertsButton,laPlaceButton,userDefButton;
    protected JButton[] pictureButtons;
    protected String[] tools={"Sobel","Roberts","LaPlace","User defined"};
    protected int gausStrength;
    protected String plaf;
    protected ImageIcon dialogIcon;
    protected DialogImageProvider imgProv;
    protected Image[] dialogImg;
    protected String[] dialogImgNames;
    protected String imageToLoad;
    
    /**
     * Initialisierung der Oberflaeche.Standardoperator
     * ist der Sobeloperator.
     */
    public void init() {
        sobel=true; 
        roberts=false;
        laPlace=false;
        userDef=false;

        imgProv = new DialogImageProvider();
        dialogImg = imgProv.getImages();
        dialogImgNames = imgProv.getImageNames();
        
        this.initializeButtons();
        this.initializeLabels();
        
 
        this.initializeDialog();
        
        Image dialogImage = Toolkit.getDefaultToolkit().getImage(getClass().getResource("dialogIcon.gif"));
        dialogIcon=new ImageIcon(dialogImage);
 
 

        gausStrength=25;
        inputImage=new iRPImage();
        inputImage.LoadImage(dialogImgNames[0]);
        outputImage=Sobel(inputImage);
        input=new ImageWindow(inputImage,10,20);
        output=new ImageWindow(outputImage,500,20);
        viewer=new ConvolutionViewer(input,output);
        
        messagePanel=new JPanel();
        messagePanel.setPreferredSize(new Dimension(230,50));
        messagePanel.setBorder(BorderFactory.createEtchedBorder());
        messagePanel.add(message1);
        messagePanel.add(message2);
        
        Dimension d = new Dimension(this.getWidth(),70);
        operationPanel = new JPanel();
        operationPanel.setLayout(new FlowLayout(FlowLayout.LEFT,10,10));
        operationPanel.setPreferredSize(d);
        operationPanel.setBorder(BorderFactory.createLineBorder(Color.GRAY));
        operationPanel.add(choosePicture);
        operationPanel.add(chooseTool);
        operationPanel.add(convolve);
        operationPanel.add(addGaus);
        operationPanel.add(saltPepper);
        operationPanel.add(messagePanel);
  
        getContentPane().add("North",operationPanel);
        getContentPane().add(viewer);
        
  
  
        this.initializeActionListeners();
        this.changeLookAndFeel();
 
        addMouseListener(this);
        addMouseMotionListener(this);
        
    }
    
    /**
     * Initialisierung der Dialogboxen,welche aufgerufen werden,
     * wenn man das Bild oder das Tool wechselt.
     */
    
    public void initializeDialog(){
        pictureDialog=new JDialog();
        
        pictureButtons=new JButton[dialogImg.length];
        String buttonName;
        
        /*
         *Hier werden die Buttons zur Bildauswahl initialisiert.
         *In 'buttonName' wird der Name des zu ladenen Bildes gespeichert.
         */
        for(int i=0;i<dialogImg.length;i++){
                        
            buttonName=dialogImgNames[i];
            pictureButtons[i]= new JButton(new ImageIcon(dialogImg[i]));
            pictureButtons[i].setBorder(BorderFactory.createBevelBorder(BevelBorder.RAISED));
            pictureButtons[i].setName(buttonName);
             
        }
  
        JOptionPane picturePane=new JOptionPane();
        picturePane.setOptions(pictureButtons);
        picturePane.setMessage("");
        pictureDialog=picturePane.createDialog(this,"select input image");

        toolMainPanel=new JPanel[1];
        toolMainPanel[0]=new JPanel();
        toolMainPanel[0].setLayout(new BorderLayout(0,0));
        
        toolSubPanel=new JPanel[2];
        toolSubPanel[0]=new JPanel();
        toolSubPanel[0].setPreferredSize(new Dimension(350,90));
        toolSubPanel[0].setLayout(new FlowLayout(FlowLayout.LEFT,10,15));
        toolSubPanel[0].add(sobelButton);
        toolSubPanel[0].add(robertsButton);
        toolSubPanel[0].add(laPlaceButton);
        toolSubPanel[0].add(userDefButton);
        
        toolSubPanel[1]=new JPanel();
        toolSubPanel[1].setPreferredSize(new Dimension(300,30));
        toolSubPanel[1].setLayout(new FlowLayout(FlowLayout.LEFT,10,5));
        toolSubPanel[1].add(new JLabel("   "));
        toolSubPanel[1].add(sobelLabel);
        toolSubPanel[1].add(new JLabel("         "));
        toolSubPanel[1].add(robertsLabel);
        toolSubPanel[1].add(new JLabel("      "));
        toolSubPanel[1].add(laPlaceLabel);
        toolSubPanel[1].add(new JLabel("  "));
        toolSubPanel[1].add(userDefLabel);
        
        toolMainPanel[0].add(BorderLayout.NORTH,toolSubPanel[0]);
        toolMainPanel[0].add(BorderLayout.CENTER,toolSubPanel[1]);
        
        JOptionPane toolPane=new JOptionPane();
        toolPane.setOptions(toolMainPanel);
        toolPane.setMessage("");
        toolDialog=toolPane.createDialog(this,"select tool");
 
    }
    
    /**
     * Initialisierung der Labels.
     * Diese werden hier im wesentlichen fuer die textuelle Beschreibung der 
     * Manipulationstools benoetigt. Die Labels message1 und message2 
     * stehen in der "Statusleiste" des Operationpanels. Sie geben immer
     * den naechsten Arbeitsschritt am, der am wahrscheinlichsten erfolgen soll. 
     */
    
    public void initializeLabels(){
 
        sobelLabel=new JLabel("Sobel");
        robertsLabel=new JLabel("Roberts");
        laPlaceLabel=new JLabel("La Place");
        userDefLabel=new JLabel("user defined");
        
        message1=new JLabel("Select a new picture/operator.You can");
        message2=new JLabel("also add some noise to the input image");
 
        
    }
    
    
    /**
     * Erstellen der Buttons.
     * Alle Buttons werden mit Icons versehen und 
     * bekommen eine Umrandung.
     */
    
    public void initializeButtons(){
        
        Image convolveImage = Toolkit.getDefaultToolkit().getImage(getClass().getResource("convolution.gif"));
        convolve = new JButton(new ImageIcon(convolveImage));
        convolve.setToolTipText("convolves the image if you have chosen the user defined operator");
        convolve.setBorder(BorderFactory.createBevelBorder(BevelBorder.RAISED));
        
        Image gaussImage=Toolkit.getDefaultToolkit().getImage(getClass().getResource("gauss.gif"));
        addGaus = new JButton(new ImageIcon(gaussImage));
        addGaus.setToolTipText("adds gaussian noise");
        addGaus.setBorder(BorderFactory.createBevelBorder(BevelBorder.RAISED));
        
        Image saltImage=Toolkit.getDefaultToolkit().getImage(getClass().getResource("salt.gif"));
        saltPepper=new JButton(new ImageIcon(saltImage));
        saltPepper.setToolTipText("adds salt & pepper");
        saltPepper.setBorder(BorderFactory.createBevelBorder(BevelBorder.RAISED));
        
        Image pictureImage=Toolkit.getDefaultToolkit().getImage(getClass().getResource("picture.gif"));
        choosePicture=new JButton(new ImageIcon(pictureImage));
        choosePicture.setToolTipText("choose picture");
        choosePicture.setBorder(BorderFactory.createBevelBorder(BevelBorder.RAISED));
        
        Image toolImage=Toolkit.getDefaultToolkit().getImage(getClass().getResource("tool.jpg"));
        ImageIcon toolIcon=new ImageIcon(toolImage);
        chooseTool=new JButton(toolIcon);
        chooseTool.setToolTipText("choose tool");
        chooseTool.setBorder(BorderFactory.createBevelBorder(BevelBorder.RAISED));   
         
        Image sobelImage=Toolkit.getDefaultToolkit().getImage(getClass().getResource("SobelDialog.jpg"));
        ImageIcon sobelIcon = new ImageIcon(sobelImage);
        sobelButton = new JButton(sobelIcon);
        sobelButton.setBorder(BorderFactory.createBevelBorder(BevelBorder.RAISED));
        
        Image robertsImage=Toolkit.getDefaultToolkit().getImage(getClass().getResource("RobertsDialog.jpg"));
        ImageIcon robertsIcon = new ImageIcon(robertsImage);
        robertsButton = new JButton(robertsIcon);
        robertsButton.setBorder(BorderFactory.createBevelBorder(BevelBorder.RAISED));
        
        Image laPlaceImage=Toolkit.getDefaultToolkit().getImage(getClass().getResource("laplaceDialog.jpg"));
        ImageIcon laPlaceIcon = new ImageIcon(laPlaceImage);
        laPlaceButton = new JButton(laPlaceIcon);
        laPlaceButton.setBorder(BorderFactory.createBevelBorder(BevelBorder.RAISED));
        
        Image userDefImage=Toolkit.getDefaultToolkit().getImage(getClass().getResource("userDefDialog.jpg"));
        ImageIcon userDefIcon = new ImageIcon(userDefImage);
        userDefButton = new JButton(userDefIcon);
        userDefButton.setBorder(BorderFactory.createBevelBorder(BevelBorder.RAISED));
    }
    
    /**
     * Mit Hilfe der ActionListener werden beim
     * Klicken auf die Buttons die entsprechenden Methoden
     * aufgerufen,z.B. wird beim druecken auf den choosePicture-button
     * die Dialogbox mit der Bildauswahl geoeffnet.
     */
    
    public void initializeActionListeners(){
        /*
         * Wechsel des Operators. 
         * Das Bild wird automatisch neu gefaltet.
         */
        choosePicture.addActionListener(new ActionListener(){
            public void actionPerformed(ActionEvent e){
                pictureDialog.show();
            }
        });
        
        
        //Hinzufuegen von gausschem Rauschen.
        
        addGaus.addActionListener(new ActionListener(){
            public void actionPerformed(ActionEvent e){
                  addGaussianNoise(inputImage,(double)gausStrength);
                  startOperation();
                  startUserDefOperation();
            }
        });
        // Hinzufuegen von salt&pepper
        saltPepper.addActionListener(new ActionListener(){
            public void actionPerformed(ActionEvent e){
                  addRandomNoise(inputImage,gausStrength*50);
                  startOperation();
                  startUserDefOperation();
            }
        });
        
        //Faltung mit eigens definiertem Kern
        
        convolve.addActionListener(new ActionListener(){
            public void actionPerformed(ActionEvent e){
                 startUserDefOperation();               
            }
        });
        
        chooseTool.addActionListener(new ActionListener(){
            public void actionPerformed(ActionEvent e){
                 toolDialog.show();            
            }
        });
        
        for (int i=0;i<dialogImg.length;i++){
            pictureButtons[i].addActionListener(new ActionListener(){
                public void actionPerformed(ActionEvent e){
                    imageToLoad=((JButton)e.getSource()).getName();
                    changePicture(imageToLoad);
                    pictureDialog.setVisible(false);
                }
            });     	
        }
        
 
        sobelButton.addActionListener(new ActionListener(){
            public void actionPerformed(ActionEvent e){
                toolDialog.setVisible(false);
                 changeOperation("Sobel");           
            }
        });
        
        robertsButton.addActionListener(new ActionListener(){
            public void actionPerformed(ActionEvent e){
                toolDialog.setVisible(false);
                 changeOperation("Roberts");           
            }
        });
        
        laPlaceButton.addActionListener(new ActionListener(){
            public void actionPerformed(ActionEvent e){
                toolDialog.setVisible(false);
                 changeOperation("LaPlace");           
            }
        });
        
        userDefButton.addActionListener(new ActionListener(){
            public void actionPerformed(ActionEvent e){
                toolDialog.setVisible(false);
                 changeOperation("User defined");           
            }
        });
    }
    
    
    /**
     * Aenderung der Optik
     *
     */
    
    public void changeLookAndFeel(){
        try{
        	plaf = "com.sun.java.swing.plaf.motif.MotifLookAndFeel";

        	UIManager.setLookAndFeel(plaf);
        	SwingUtilities.updateComponentTreeUI(this);
        	SwingUtilities.updateComponentTreeUI(operationPanel);
        	SwingUtilities.updateComponentTreeUI(choosePicture);
        	SwingUtilities.updateComponentTreeUI(addGaus);
        	SwingUtilities.updateComponentTreeUI(saltPepper);
        	SwingUtilities.updateComponentTreeUI(chooseTool);
           	SwingUtilities.updateComponentTreeUI(convolve);
          	SwingUtilities.updateComponentTreeUI(sobelButton);
           	SwingUtilities.updateComponentTreeUI(robertsButton);
           	SwingUtilities.updateComponentTreeUI(laPlaceButton);
           	SwingUtilities.updateComponentTreeUI(userDefButton);
           	
       		for (int i=0;i<pictureButtons.length;i++){
    		    SwingUtilities.updateComponentTreeUI(pictureButtons[i]);
    		}
 
             } catch (UnsupportedLookAndFeelException e) {
               System.err.println(e.toString());
             } catch (ClassNotFoundException e) {
               System.err.println(e.toString());
             } catch (InstantiationException e) {
               System.err.println(e.toString());
             } catch (IllegalAccessException e) {
               System.err.println(e.toString());
             }
    }
    
    /**
     * Die Methode realisert den Bildwechsel.
     * Sowohl das Bild selbst,als auch das ImageWindow-Objekt wird
     * aktualisiert.Anschlieend wird startOperation() aufgerufen,
     * um das neue Bild gleich zu falten.
     * @param pic Name des Bildes
     */
    public void changePicture(String pic){
             
        inputImage.LoadImage(pic);
        input.changeImage(inputImage);
        
        startOperation();
        startUserDefOperation();
     
    }
    
    /**
     * In dieser Methode wird der Operator gewechselt,abhaengig von 'op'.
     * Dann wird startOperation() aufgerufen,um die neue Faltung durch
     * zu fuehren.
     * @param op Name des Operators
     */
    public void changeOperation(String op){
        
 
        
        if(op=="Sobel"){
            sobel=true;
            laPlace=false;
            roberts=false;
            userDef=false;
        }
        if(op=="Roberts"){
            sobel=false;
            laPlace=false;
            roberts=true;
            userDef=false;
        }
        if(op=="LaPlace"){
            sobel=false;
            laPlace=true;
            roberts=false;
            userDef=false;
        }
        
        if(op=="User defined"){
            sobel=false;
            laPlace=false;
            roberts=false;
            userDef=true;
 
        }
        
        if(op=="User defined"){
            message1.setText("Define convolution function and");
            message2.setText("press button with convolution symbol");
        }
        else{
            message1.setText("Select a new picture/operator.You can");
            message2.setText("also add some noise to the input image");
        }
        
        startOperation();
        startUserDefOperation();
    }
    
    /**
     * Startet abhaengig von den boolschen Werten
     * die entsprechende Faltung. Auch die Variablen im
     * ConvolutionViewer werden zwecks optischer Darstellung
     * neu definiert.
     */
    public void startOperation() {
        if(sobel){
            outputImage=Sobel(inputImage);
            viewer.setSobel(true);
            viewer.setRoberts(false);
            viewer.setLaPlace(false);
            viewer.setUserDef(false);
        }
        else if(roberts){
            outputImage=Roberts(inputImage);
            viewer.setSobel(false);
            viewer.setRoberts(true);
            viewer.setLaPlace(false);
            viewer.setUserDef(false);
        }
        else if(laPlace){
            outputImage= Laplace(inputImage);
            viewer.setSobel(false);
            viewer.setRoberts(false);
            viewer.setLaPlace(true);
            viewer.setUserDef(false);
        }
        else if(userDef){
            outputImage=inputImage.copy();
            viewer.setSobel(false);
            viewer.setRoberts(false);
            viewer.setLaPlace(false);
            viewer.setUserDef(true);
        }
        
        output.changeImage(outputImage);
        repaint();
    }
    
    /**
     * Startet die Faltung mit dem benutzerdefiniertem
     * Faltungskern.
     */
    public void startUserDefOperation() {
        if(userDef){
            iRPImage convMask=viewer.getMask();
            outputImage= Convolve(inputImage,convMask);
            output.changeImage(outputImage);
            repaint();
        }
    }
    
    /**
     * Fuegt gaussches Rauschen zu dem Eingangsbild hinzu.
     * @param A das Eingangsbild.
     * @param strength Staerke des Rauschens.
     */
    
    public void addGaussianNoise(iRPImage A,double strength)
    {
    	Random randomizer = new Random();

    	for(int y=0; y<A.GetSizeY(); y++)
    	{
    		for(int x=0; x<A.GetSizeX(); x++)
    		{
    			A.SetPixel(x,y,A.GetPixel(x,y)+(int)(strength*randomizer.nextGaussian()));
    		}
    	}

    }
    
    /**
     * Hinzufuegen von salt&pepper
     * @param A Bild,in welches die Stoerung hinzugefuegt werden soll
     * @param intensity die Staerke der Stoerung.
     */
    public void addRandomNoise(iRPImage A, int intensity)
    {
    	Random randomizer = new Random();
    	int randomNumber1, randomNumber2;
    	boolean black;

    	for(int y=0; y<intensity; y++)
    	{
    		randomNumber1 = Math.abs(randomizer.nextInt()%256);
    		randomNumber2 = Math.abs(randomizer.nextInt()%256);
    		black = randomizer.nextBoolean();
    		if(black)
    		{
    			A.SetPixel(randomNumber1, randomNumber2, 0);
    		}
    		else
    		{
    			A.SetPixel(randomNumber1, randomNumber2, 255);
    		}
    	}
    }
    
    /**
     * Realisiert den Sobeloperator.
     * Bei einen 3*3 Operatorfenster
     * ABC
     * DEF
     * GHI
     * ist dieser Operator definiert durch:
     * GSobel=|(A+2*B+C)-(G+2*H+I)|+|(A+2*D+G)-(C+2*F+I)|
     * @param A das Eingangsbild
     */
	protected iRPImage Sobel(iRPImage A){

		int SizeX = A.GetSizeX();
		int SizeY = A.GetSizeY();
		iRPImage B = new iRPImage(SizeX,SizeY);
		int res1, res2, d1, d2;

			for(int c1=0 , c2=1, c3 = 2,
		        c4 = SizeX, c5= SizeX+1,c6=SizeX+2,
				c7=SizeX*2, c8=(SizeX*2)+1, c9=(SizeX*2)+2
				; c1 < SizeX*(SizeY-2)-3
				;c1++,c2++,c3++,c4++,c5++,c6++,c7++,c8++,c9++)
			{
				d1 = A.GetPixel(c1) + 2 * (A.GetPixel(c4)) + A.GetPixel(c7);
				d2 = A.GetPixel(c3) + 2 * (A.GetPixel(c6)) + A.GetPixel(c9);
				res1 = d1 < d2 ?  d2-d1 : d1-d2;
				d1 = A.GetPixel(c1) + 2 * (A.GetPixel(c2)) + A.GetPixel(c3);
				d2 = A.GetPixel(c7) + 2 * (A.GetPixel(c8)) + A.GetPixel(c9);
				res2 = d1 < d2 ?  d2-d1 : d1-d2;
				B.SetPixel(c5, res1+res2);
			}
			return B;
	}

	/**
	 * Realisiert den Roberts-Gradient:
	 * Bei einem 3*3 Operatorfenster
	 * ABC
	 * DEF
	 * GHI
	 * ist dieser definiert durch:
	 * GRoberts=max(|A-E|,|B-D|)
	 * @param A das Eingangsbild
	 */
	protected iRPImage Roberts(iRPImage A){

		int SizeX = A.GetSizeX();
		int SizeY = A.GetSizeY();
		iRPImage B = new iRPImage(SizeX,SizeY);
		int sum1, sum2;

			for(int c1=0, c2=1 ,c3=SizeX, c4=SizeX+1 ;
				c1 < SizeX*(SizeY-2) ;
				c1++,c2++,c3++,c4++)
			{
				sum1 =  A.GetPixel(c4) < A.GetPixel(c1) ?
					  	A.GetPixel(c1) - A.GetPixel(c4) :
					  	A.GetPixel(c4) - A.GetPixel(c1) ;

				sum2 =  A.GetPixel(c2) < A.GetPixel(c3) ?
						A.GetPixel(c3) - A.GetPixel(c2) :
						A.GetPixel(c2) - A.GetPixel(c3) ;
				
	
				if (sum1>sum2){
					B.SetPixel(c1, sum1);
				}
				else {
					B.SetPixel(c1,sum2);
				}

		
			}

			return B;

	}

	/**
	 * Realisiert den LaPlace-Operator.
	 * Bei einem 3*3 Operatorfenster 
	 * ABC
	 * DEF
	 * GHI
	 * ist dieser definiert durch
	 * GLaPlace=|B+D+H+F-4*E|
	 * @param A das Eingangsbild
	 */
	protected iRPImage Laplace(iRPImage A){

		int SizeX = A.GetSizeX();
		int SizeY = A.GetSizeY();
		iRPImage B = new iRPImage(SizeX, SizeY);
		int sum;

			for(int c1=0, c2=1, c3=3,
					c4 = SizeX, c5= SizeX+1,c6=SizeX+2,
					c7=SizeX*2, c8=(SizeX*2)+1, c9=(SizeX*2)+2
					; c1 < SizeX*(SizeY-2)-3
				    ; c1++,c2++,c3++,c4++,c5++,c6++,c7++,c8++,c9++)
			{

				sum = A.GetPixel(c2) + A.GetPixel(c4) + A.GetPixel(c8)
					+ A.GetPixel(c6) - 4 * A.GetPixel(c5);
				if(sum < 0) sum = -(sum);

				B.SetPixel(c5, sum);
			}

			return B;

	}

	/**
	 * Diese Methode faltet ein Eingangsbild A mit einem 
	 * selbst definiertem Faltungskern. Dieser ist in dem 
	 * Bild B kodiert. Bei einem 3*3 Operatorfenster haette B
	 * dementsprechend 9 Pixel;
	 * @param A das Eingangsbild
	 * @param B der Faltungskern als Bild dargestellt
	 */ 
	protected iRPImage Convolve(iRPImage A, iRPImage B) {

		int a, newX, newY;
		int n, m;
		n = B.GetSizeX()/2;
		m = B.GetSizeY()/2;
		// iRPImage c wird erstellt
		iRPImage C = new iRPImage();
		C.SetSize(A.GetSizeX(), A.GetSizeY());

		for(int y = 0; y < A.GetSizeY() ; y++) {
			for(int x = 0; x < A.GetSizeX() ; x++){

				a = A.GetPixel(x, y);
				for (int j=0; j < B.GetSizeY() ; j++) {
					for(int i=0; i < B.GetSizeX(); i++){

					newX = x - n + i;
					newY = y - m + j;
					if( (newX > 0) && (newY > 0) && (newX < A.GetSizeX()) && (newY < A.GetSizeY()) )
					  C.SetPixel(newX, newY, (B.GetPixel(i,j) * a) + C.GetPixel(newX,newY));
					else;

						} // j-Schleife
					} //i Schleife
				} // y-for zu ende
			}  // x-for zu ende


		return C;

	}




	/**
	 * Die Methode setzt abhaengig von der Mausposition
	 * die Grauwerte und die Mauskoordinaten fuer die Anzeige
	 * unterhalb des Bildes neu.
	 */
    public void mouseMoved(MouseEvent e) {
        int xPos=e.getX();
        int yPos=e.getY();
        int value;
        if(xPos>=10 && xPos<266 && yPos>=90 && yPos<346){
            value=inputImage.GetPixel(xPos-10,yPos-90);
            input.updateMousePosition(xPos-10,yPos-90,value);
        }
        else if(xPos>=500 && xPos<756 && yPos>=90 && yPos<346){
            value=outputImage.GetPixel(xPos-500,yPos-90);
            output.updateMousePosition(xPos-500,yPos-90,value);
        }
        repaint();
        
        
    }
    
    
    public void mouseClicked(MouseEvent e) {
        // TODO Auto-generated method stub
        
    }

    public void mouseEntered(MouseEvent e) {
        // TODO Auto-generated method stub
        
    }

    public void mouseExited(MouseEvent e) {
        // TODO Auto-generated method stub
        
    }

    public void mousePressed(MouseEvent e) {
        // TODO Auto-generated method stub
        
    }

    public void mouseReleased(MouseEvent e) {
        // TODO Auto-generated method stub
        
    }

    public void mouseDragged(MouseEvent e) {
        // TODO Auto-generated method stub
        
    }


    
    

}
